import { useState, useEffect } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import api from '../../services/api';
import Loading from '../Common/Loading';
import QuestionCard from './QuestionCard';
import { toast } from 'react-toastify';
import { FiArrowRight, FiCheck } from 'react-icons/fi';

export default function QuizTake() {
  const { id } = useParams();
  const navigate = useNavigate();
  const [quiz, setQuiz] = useState(null);
  const [attemptId, setAttemptId] = useState(null);
  const [currentIndex, setCurrentIndex] = useState(0);
  const [selectedAnswer, setSelectedAnswer] = useState(null);
  const [showResult, setShowResult] = useState(false);
  const [answerResult, setAnswerResult] = useState(null);
  const [loading, setLoading] = useState(true);
  const [submitting, setSubmitting] = useState(false);
  const [startTime] = useState(Date.now());

  useEffect(() => {
    const loadQuiz = async () => {
      try {
        const quizRes = await api.get(`/quiz/${id}`);
        setQuiz(quizRes.data);
        const startRes = await api.post(`/quiz/${id}/start`);
        setAttemptId(startRes.data.attemptId);
      } catch (error) {
        toast.error('Failed to load quiz');
        navigate('/');
      } finally {
        setLoading(false);
      }
    };
    loadQuiz();
  }, [id, navigate]);

  const handleSubmitAnswer = async () => {
    if (!selectedAnswer) {
      toast.warning('Please select an answer');
      return;
    }
    setSubmitting(true);
    try {
      const scenario = quiz.scenarios[currentIndex];
      const res = await api.post('/quiz/answer', { attemptId, scenarioId: scenario.id, selectedAnswer });
      setAnswerResult(res.data);
      setShowResult(true);
    } catch (error) {
      toast.error('Failed to submit answer');
    } finally {
      setSubmitting(false);
    }
  };

  const handleNext = async () => {
    if (currentIndex < quiz.scenarios.length - 1) {
      setCurrentIndex(currentIndex + 1);
      setSelectedAnswer(null);
      setShowResult(false);
      setAnswerResult(null);
    } else {
      const timeTaken = Math.round((Date.now() - startTime) / 1000);
      try {
        await api.post('/quiz/complete', { attemptId, timeTaken });
        navigate(`/result/${attemptId}`);
      } catch (error) {
        toast.error('Failed to complete quiz');
      }
    }
  };

  if (loading) return <Loading />;
  if (!quiz) return null;

  const currentScenario = quiz.scenarios[currentIndex];
  const progress = ((currentIndex + 1) / quiz.scenarios.length) * 100;

  return (
    <div className="max-w-4xl mx-auto">
      <div className="bg-white rounded-xl shadow-lg p-6 mb-6">
        <div className="flex justify-between items-center mb-4">
          <h1 className="text-2xl font-bold text-gray-800">{quiz.title}</h1>
          <span className="text-gray-500">Question {currentIndex + 1} of {quiz.scenarios.length}</span>
        </div>
        <div className="w-full bg-gray-200 rounded-full h-2">
          <div className="bg-indigo-600 h-2 rounded-full transition-all duration-300" style={{ width: `${progress}%` }}></div>
        </div>
      </div>
      <QuestionCard scenario={currentScenario} selectedAnswer={selectedAnswer} onSelectAnswer={setSelectedAnswer} showResult={showResult} answerResult={answerResult} />
      <div className="mt-6 flex justify-end">
        {!showResult ? (
          <button onClick={handleSubmitAnswer} disabled={submitting || !selectedAnswer} className="bg-indigo-600 text-white px-6 py-3 rounded-lg font-semibold hover:bg-indigo-700 transition flex items-center gap-2 disabled:opacity-50">
            {submitting ? 'Submitting...' : <><FiCheck /> Submit Answer</>}
          </button>
        ) : (
          <button onClick={handleNext} className="bg-green-600 text-white px-6 py-3 rounded-lg font-semibold hover:bg-green-700 transition flex items-center gap-2">
            {currentIndex < quiz.scenarios.length - 1 ? <><FiArrowRight /> Next Question</> : <><FiCheck /> Finish Quiz</>}
          </button>
        )}
      </div>
    </div>
  );
}