// src/models/User.js
const db = require("../config/database");
const bcrypt = require("bcrypt");

class User {
  static async create({ username, email, password }) {
    const saltRounds = 12;
    const passwordHash = await bcrypt.hash(password, saltRounds);

    const query = `
            INSERT INTO users (username, email, password_hash)
            VALUES ($1, $2, $3)
            RETURNING id, username, email, created_at
        `;

    const result = await db.query(query, [username, email, passwordHash]);
    return result.rows[0];
  }

  static async findByEmail(email) {
    const query = "SELECT * FROM users WHERE email = $1 AND is_active = true";
    const result = await db.query(query, [email]);
    return result.rows[0];
  }

  static async findByUsername(username) {
    const query =
      "SELECT * FROM users WHERE username = $1 AND is_active = true";
    const result = await db.query(query, [username]);
    return result.rows[0];
  }

  static async findById(id) {
    const query = `
            SELECT id, username, email, created_at, updated_at 
            FROM users 
            WHERE id = $1 AND is_active = true
        `;
    const result = await db.query(query, [id]);
    return result.rows[0];
  }

  static async verifyPassword(plainPassword, hashedPassword) {
    return bcrypt.compare(plainPassword, hashedPassword);
  }

  static async getStatistics(userId) {
    const query = "SELECT * FROM user_statistics WHERE user_id = $1";
    const result = await db.query(query, [userId]);
    return result.rows[0];
  }

  static async saveRefreshToken(
    userId,
    tokenHash,
    expiresAt,
    ipAddress,
    userAgent
  ) {
    const query = `
            INSERT INTO user_sessions (user_id, refresh_token_hash, expires_at, ip_address, user_agent)
            VALUES ($1, $2, $3, $4, $5)
            RETURNING id
        `;
    const result = await db.query(query, [
      userId,
      tokenHash,
      expiresAt,
      ipAddress,
      userAgent,
    ]);
    return result.rows[0];
  }

  static async findRefreshToken(tokenHash) {
    const query = `
            SELECT * FROM user_sessions 
            WHERE refresh_token_hash = $1 
            AND is_revoked = false 
            AND expires_at > NOW()
        `;
    const result = await db.query(query, [tokenHash]);
    return result.rows[0];
  }

  static async revokeRefreshToken(tokenHash) {
    const query = `
            UPDATE user_sessions 
            SET is_revoked = true 
            WHERE refresh_token_hash = $1
        `;
    await db.query(query, [tokenHash]);
  }

  static async revokeAllUserTokens(userId) {
    const query = `
            UPDATE user_sessions 
            SET is_revoked = true 
            WHERE user_id = $1
        `;
    await db.query(query, [userId]);
  }
}

module.exports = User;
