// src/controllers/scenarioController.js
const Scenario = require("../models/Scenario");
const Progress = require("../models/Progress");

const getScenarioById = async (req, res, next) => {
  try {
    const { id } = req.params;
    const scenario = await Scenario.findById(id);

    if (!scenario) {
      return res.status(404).json({
        success: false,
        message: "Scenario not found",
      });
    }

    // Check if user has completed this scenario
    let userProgress = null;
    if (req.user) {
      const progressRecords = await Progress.findByUserAndScenario(
        req.user.userId,
        id
      );
      if (progressRecords.length > 0) {
        userProgress = {
          attempts: progressRecords.length,
          bestScore: Math.max(...progressRecords.map((p) => p.score)),
          lastAttempt: progressRecords[0],
        };
      }
    }

    // Prepare response - shuffle answers
    const allAnswers = [
      { ...scenario.best_answer, id: "correct" },
      ...scenario.other_answers.map((ans, idx) => ({
        ...ans,
        id: `wrong_${idx}`,
      })),
    ];

    // Fisher-Yates shuffle
    for (let i = allAnswers.length - 1; i > 0; i--) {
      const j = Math.floor(Math.random() * (i + 1));
      [allAnswers[i], allAnswers[j]] = [allAnswers[j], allAnswers[i]];
    }

    // Remove rationale/explanation from response (will show after answer)
    const sanitizedAnswers = allAnswers.map((ans) => ({
      id: ans.id,
      answer: ans.answer,
    }));

    res.json({
      success: true,
      data: {
        scenario: {
          id: scenario.id,
          title: scenario.title,
          shortDescription: scenario.short_description,
          givensTable: scenario.givens_table,
          scenarioParagraph: scenario.scenario_paragraph,
          difficultyLevel: scenario.difficulty_level,
          category: scenario.category,
          tags: scenario.tags,
        },
        answers: sanitizedAnswers,
        userProgress,
      },
    });
  } catch (error) {
    next(error);
  }
};

const getAllScenarios = async (req, res, next) => {
  try {
    const { limit = 10, offset = 0, category, difficulty } = req.query;

    const result = await Scenario.findAll({
      limit: parseInt(limit),
      offset: parseInt(offset),
      category,
      difficulty,
    });

    // If user is authenticated, add completion status
    if (req.user) {
      const completedIds = await Progress.getCompletedScenarioIds(
        req.user.userId
      );
      result.scenarios = result.scenarios.map((scenario) => ({
        ...scenario,
        isCompleted: completedIds.includes(scenario.id),
      }));
    }

    res.json({
      success: true,
      data: result,
    });
  } catch (error) {
    next(error);
  }
};

const getCategories = async (req, res, next) => {
  try {
    const categories = await Scenario.getCategories();
    res.json({
      success: true,
      data: categories,
    });
  } catch (error) {
    next(error);
  }
};

const getScenarioStatistics = async (req, res, next) => {
  try {
    const { id } = req.params;
    const statistics = await Scenario.getStatistics(id);

    if (!statistics) {
      return res.status(404).json({
        success: false,
        message: "Scenario not found",
      });
    }

    res.json({
      success: true,
      data: statistics,
    });
  } catch (error) {
    next(error);
  }
};

module.exports = {
  getScenarioById,
  getAllScenarios,
  getCategories,
  getScenarioStatistics,
};
