// src/controllers/progressController.js
const Progress = require("../models/Progress");
const Scenario = require("../models/Scenario");

const submitAnswer = async (req, res, next) => {
  try {
    const { scenarioId, selectedAnswer, timeSpentSeconds } = req.body;
    const userId = req.user.userId;

    // Get scenario to verify answer
    const scenario = await Scenario.findById(scenarioId);
    if (!scenario) {
      return res.status(404).json({
        success: false,
        message: "Scenario not found",
      });
    }

    // Check if answer is correct
    const isCorrect = scenario.best_answer.answer === selectedAnswer;

    // Calculate score (base score + time bonus)
    let score = isCorrect ? 100 : 0;
    if (isCorrect && timeSpentSeconds) {
      // Time bonus: up to 20 extra points for fast answers
      const timeBonus = Math.max(0, 20 - Math.floor(timeSpentSeconds / 30));
      score += score >= 100 ? 0 : timeBonus;
    }

    // Save progress
    console.log("Score => ", score);
    const progress = await Progress.create({
      userId,
      scenarioId,
      selectedAnswer,
      isCorrect,
      score,
      timeSpentSeconds,
    });

    // Get feedback
    let feedback;
    if (isCorrect) {
      console.log("isCorrect=>", progress);
      feedback = {
        correct: true,
        rationale: scenario.best_answer.rationale,
      };
    } else {
      // Find the selected wrong answer
      const wrongAnswer = scenario.other_answers.find(
        (ans) => ans.answer === selectedAnswer
      );
      feedback = {
        correct: false,
        explanation:
          wrongAnswer?.explanation || "This was not the best answer.",
        correctAnswer: scenario.best_answer.answer,
        rationale: scenario.best_answer.rationale,
      };
    }

    res.json({
      success: true,
      data: {
        progress: {
          id: progress.id,
          isCorrect: progress.is_correct,
          score: progress.score,
          attemptNumber: progress.attempt_number,
        },
        feedback,
      },
    });
  } catch (error) {
    next(error);
  }
};

const getUserProgress = async (req, res, next) => {
  try {
    const { limit = 10, offset = 0 } = req.query;
    const userId = req.user.userId;

    const progress = await Progress.getUserProgress(userId, {
      limit: parseInt(limit),
      offset: parseInt(offset),
    });

    res.json({
      success: true,
      data: progress,
    });
  } catch (error) {
    next(error);
  }
};

const getLeaderboard = async (req, res, next) => {
  try {
    const { limit = 10 } = req.query;
    const leaderboard = await Progress.getLeaderboard(parseInt(limit));

    res.json({
      success: true,
      data: leaderboard,
    });
  } catch (error) {
    next(error);
  }
};

module.exports = {
  submitAnswer,
  getUserProgress,
  getLeaderboard,
};
