import { useState, useEffect } from 'react';
import { useParams, Link } from 'react-router-dom';
import api from '../../services/api';
import Loading from '../Common/Loading';
import { FiCheckCircle, FiXCircle, FiClock, FiHome, FiRepeat } from 'react-icons/fi';

export default function QuizResult() {
  const { attemptId } = useParams();
  const [result, setResult] = useState(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    api.get(`/quiz/result/${attemptId}`).then(res => setResult(res.data)).finally(() => setLoading(false));
  }, [attemptId]);

  if (loading) return <Loading />;
  if (!result) return null;

  const passed = result.score >= 60;

  return (
    <div className="max-w-2xl mx-auto">
      <div className="bg-white rounded-xl shadow-lg overflow-hidden">
        <div className={`p-8 text-center ${passed ? 'bg-green-500' : 'bg-red-500'}`}>
          {passed ? <FiCheckCircle className="mx-auto text-white" size={64} /> : <FiXCircle className="mx-auto text-white" size={64} />}
          <h1 className="text-3xl font-bold text-white mt-4">{passed ? 'Congratulations!' : 'Keep Practicing!'}</h1>
          <p className="text-white opacity-90 mt-2">{passed ? 'You passed the quiz!' : 'You did not pass this time.'}</p>
        </div>
        <div className="p-8">
          <h2 className="text-2xl font-bold text-gray-800 text-center mb-6">{result.quiz_title}</h2>
          <div className="grid grid-cols-3 gap-6 mb-8">
            <div className="text-center">
              <div className="text-4xl font-bold text-indigo-600">{result.score}%</div>
              <div className="text-gray-500 mt-1">Score</div>
            </div>
            <div className="text-center">
              <div className="text-4xl font-bold text-green-600">{result.correct_answers}/{result.total_questions}</div>
              <div className="text-gray-500 mt-1">Correct</div>
            </div>
            <div className="text-center">
              <div className="text-4xl font-bold text-gray-600 flex items-center justify-center gap-1">
                <FiClock /> {Math.floor(result.time_taken / 60)}:{String(result.time_taken % 60).padStart(2, '0')}
              </div>
              <div className="text-gray-500 mt-1">Time</div>
            </div>
          </div>
          <div className="flex justify-center gap-4">
            <Link to="/" className="flex items-center gap-2 bg-indigo-600 text-white px-6 py-3 rounded-lg font-semibold hover:bg-indigo-700 transition">
              <FiHome /> Back to Quizzes
            </Link>
            <Link to={`/quiz/${result.quiz_id}`} className="flex items-center gap-2 border-2 border-indigo-600 text-indigo-600 px-6 py-3 rounded-lg font-semibold hover:bg-indigo-50 transition">
              <FiRepeat /> Try Again
            </Link>
          </div>
        </div>
      </div>
    </div>
  );
}