import { useState, useEffect } from 'react';
import api from '../../services/api';
import Loading from '../Common/Loading';
import Modal from '../Common/Modal';
import { toast } from 'react-toastify';
import { FiPlus, FiEdit2, FiTrash2, FiSave } from 'react-icons/fi';

const emptyQuiz = { title: '', description: '', time_limit: 0, passing_score: 60, scenario_ids: [] };

export default function QuizManager() {
  const [quizzes, setQuizzes] = useState([]);
  const [scenarios, setScenarios] = useState([]);
  const [loading, setLoading] = useState(true);
  const [editQuiz, setEditQuiz] = useState(null);
  const [showModal, setShowModal] = useState(false);
  const [isNew, setIsNew] = useState(false);

  const loadData = async () => {
    const [quizzesRes, scenariosRes] = await Promise.all([api.get('/admin/quizzes'), api.get('/admin/scenarios')]);
    setQuizzes(quizzesRes.data);
    setScenarios(scenariosRes.data);
    setLoading(false);
  };

  useEffect(() => { loadData(); }, []);

  const handleSubmit = async (e) => {
    e.preventDefault();
    try {
      if (isNew) {
        await api.post('/admin/quizzes', editQuiz);
        toast.success('Quiz created successfully');
      } else {
        await api.put(`/admin/quizzes/${editQuiz.id}`, editQuiz);
        toast.success('Quiz updated successfully');
      }
      setShowModal(false);
      loadData();
    } catch (error) {
      toast.error(error.response?.data?.message || 'Operation failed');
    }
  };

  const handleDelete = async (id) => {
    if (!confirm('Are you sure you want to delete this quiz?')) return;
    try {
      await api.delete(`/admin/quizzes/${id}`);
      toast.success('Quiz deleted successfully');
      loadData();
    } catch (error) {
      toast.error(error.response?.data?.message || 'Delete failed');
    }
  };

  const toggleScenario = (scenarioId) => {
    const ids = editQuiz.scenario_ids || [];
    if (ids.includes(scenarioId)) {
      setEditQuiz({ ...editQuiz, scenario_ids: ids.filter(id => id !== scenarioId) });
    } else {
      setEditQuiz({ ...editQuiz, scenario_ids: [...ids, scenarioId] });
    }
  };

  if (loading) return <Loading />;

  return (
    <div>
      <div className="flex justify-between items-center mb-8">
        <h1 className="text-3xl font-bold text-gray-800">Manage Quizzes</h1>
        <button onClick={() => { setEditQuiz(emptyQuiz); setIsNew(true); setShowModal(true); }} className="flex items-center gap-2 bg-indigo-600 text-white px-4 py-2 rounded-lg hover:bg-indigo-700">
          <FiPlus /> Add Quiz
        </button>
      </div>
      <div className="bg-white rounded-xl shadow-lg overflow-hidden">
        <table className="w-full">
          <thead className="bg-gray-50">
            <tr>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Title</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Questions</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Time Limit</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Pass Score</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Actions</th>
            </tr>
          </thead>
          <tbody className="divide-y divide-gray-200">
            {quizzes.map(quiz => (
              <tr key={quiz.id}>
                <td className="px-6 py-4 font-medium text-gray-800">{quiz.title}</td>
                <td className="px-6 py-4 text-gray-600">{quiz.question_count}</td>
                <td className="px-6 py-4 text-gray-600">{quiz.time_limit > 0 ? `${quiz.time_limit} min` : 'No limit'}</td>
                <td className="px-6 py-4 text-gray-600">{quiz.passing_score}%</td>
                <td className="px-6 py-4">
                  <span className={`px-2 py-1 rounded text-sm ${quiz.is_active ? 'bg-green-100 text-green-600' : 'bg-gray-100 text-gray-600'}`}>
                    {quiz.is_active ? 'Active' : 'Inactive'}
                  </span>
                </td>
                <td className="px-6 py-4">
                  <button onClick={() => { setEditQuiz({ ...quiz, scenario_ids: [] }); setIsNew(false); setShowModal(true); }} className="text-indigo-600 hover:text-indigo-800 mr-3"><FiEdit2 /></button>
                  <button onClick={() => handleDelete(quiz.id)} className="text-red-600 hover:text-red-800"><FiTrash2 /></button>
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
      <Modal isOpen={showModal} onClose={() => setShowModal(false)} title={isNew ? 'Add Quiz' : 'Edit Quiz'}>
        {editQuiz && (
          <form onSubmit={handleSubmit} className="space-y-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">Title</label>
              <input type="text" value={editQuiz.title} onChange={(e) => setEditQuiz({ ...editQuiz, title: e.target.value })} className="w-full px-4 py-2 border border-gray-300 rounded-lg" required />
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">Description</label>
              <textarea value={editQuiz.description || ''} onChange={(e) => setEditQuiz({ ...editQuiz, description: e.target.value })} className="w-full px-4 py-2 border border-gray-300 rounded-lg" rows="3" />
            </div>
            <div className="grid grid-cols-2 gap-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Time Limit (minutes, 0 for none)</label>
                <input type="number" value={editQuiz.time_limit} onChange={(e) => setEditQuiz({ ...editQuiz, time_limit: parseInt(e.target.value) })} className="w-full px-4 py-2 border border-gray-300 rounded-lg" min="0" />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Passing Score (%)</label>
                <input type="number" value={editQuiz.passing_score} onChange={(e) => setEditQuiz({ ...editQuiz, passing_score: parseInt(e.target.value) })} className="w-full px-4 py-2 border border-gray-300 rounded-lg" min="0" max="100" />
              </div>
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Select Scenarios ({editQuiz.scenario_ids?.length || 0} selected)</label>
              <div className="max-h-60 overflow-y-auto border border-gray-300 rounded-lg p-2">
                {scenarios.map(scenario => (
                  <label key={scenario.id} className="flex items-center gap-2 p-2 hover:bg-gray-50 cursor-pointer">
                    <input type="checkbox" checked={editQuiz.scenario_ids?.includes(scenario.id)} onChange={() => toggleScenario(scenario.id)} className="rounded text-indigo-600" />
                    <span className="text-sm text-gray-700">{scenario.title}</span>
                    <span className={`ml-auto text-xs px-2 py-1 rounded ${scenario.event_type === 'major' ? 'bg-red-100 text-red-600' : 'bg-blue-100 text-blue-600'}`}>
                      {scenario.event_type}
                    </span>
                  </label>
                ))}
              </div>
            </div>
            <button type="submit" className="flex items-center gap-2 bg-indigo-600 text-white px-4 py-2 rounded-lg hover:bg-indigo-700">
              <FiSave /> {isNew ? 'Create Quiz' : 'Save Changes'}
            </button>
          </form>
        )}
      </Modal>
    </div>
  );
}