const pool = require("../config/database");

exports.getAllQuizzes = async (req, res) => {
  try {
    const [quizzes] = await pool.execute(`
            SELECT q.*, u.username as creator_name, 
            (SELECT COUNT(*) FROM quiz_scenarios WHERE quiz_id = q.id) as question_count
            FROM quizzes q 
            LEFT JOIN users u ON q.created_by = u.id 
            WHERE q.is_active = TRUE
            ORDER BY q.created_at DESC
        `);
    res.json(quizzes);
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

exports.getQuizById = async (req, res) => {
  try {
    const [quizzes] = await pool.execute("SELECT * FROM quizzes WHERE id = ?", [
      req.params.id,
    ]);
    if (quizzes.length === 0) {
      return res.status(404).json({ message: "Quiz not found" });
    }
    const [scenarios] = await pool.execute(
      `
            SELECT s.*, qs.question_order 
            FROM scenarios s 
            JOIN quiz_scenarios qs ON s.id = qs.scenario_id 
            WHERE qs.quiz_id = ? AND s.is_active = TRUE
            ORDER BY qs.question_order
        `,
      [req.params.id]
    );
    res.json({ ...quizzes[0], scenarios });
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

exports.startQuiz = async (req, res) => {
  try {
    // console.log("Start Quiz");
    const [existing] = await pool.execute(
      'SELECT * FROM quiz_attempts WHERE user_id = ? AND quiz_id = ? AND status = "in_progress"',
      [req.user.id, req.params.id]
    );
    if (existing.length > 0) {
      // console.log("existing", existing);
      return res.json({
        attemptId: existing[0].id,
        message: "Continuing existing attempt",
      });
    }
    const [scenarios] = await pool.execute(
      "SELECT COUNT(*) as count FROM quiz_scenarios WHERE quiz_id = ?",
      [req.params.id]
    );
    // console.log("secenarios", scenarios[0].count);
    const [result] = await pool.execute(
      "INSERT INTO quiz_attempts (user_id, quiz_id, total_questions) VALUES (?, ?, ?)",
      [req.user.id, req.params.id, scenarios[0].count]
    );
    res.json({ attemptId: result.insertId });
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

exports.submitAnswer = async (req, res) => {
  try {
    const { attemptId, scenarioId, selectedAnswer } = req.body;
    // console.log("req.body", req.body);
    const [scenarios] = await pool.execute(
      "SELECT * FROM scenarios WHERE id = ?",
      [scenarioId]
    );
    if (scenarios.length === 0) {
      return res.status(404).json({ message: "Scenario not found" });
    }
    const scenario = scenarios[0];
    const isCorrect = selectedAnswer === scenario.best_answer;
    await pool.execute(
      "INSERT INTO user_scenario_history (user_id, scenario_id, selected_answer, is_correct, attempt_id) VALUES (?, ?, ?, ?, ?)",
      [req.user.id, scenarioId, selectedAnswer, isCorrect, attemptId]
    );
    const [attempt] = await pool.execute(
      "SELECT answers FROM quiz_attempts WHERE id = ?",
      [attemptId]
    );

    let answers = attempt[0].answers
      ? typeof attempt[0].answers === "string"
        ? JSON.parse(attempt[0].answers)
        : attempt[0].answers
      : {};
    answers[scenarioId] = { selectedAnswer, isCorrect };
    const correctCount = Object.values(answers).filter(
      (a) => a.isCorrect
    ).length;
    await pool.execute(
      "UPDATE quiz_attempts SET answers = ?, correct_answers = ? WHERE id = ?",
      [JSON.stringify(answers), correctCount, attemptId]
    );
    res.json({
      isCorrect,
      correctAnswer: scenario.best_answer,
      rationale: scenario.best_answer_rationale,
    });
  } catch (error) {
    console.log("error", error);
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

exports.completeQuiz = async (req, res) => {
  try {
    const { attemptId, timeTaken } = req.body;
    // console.log("req.body", req.body);
    const [attempts] = await pool.execute(
      "SELECT * FROM quiz_attempts WHERE id = ? AND user_id = ?",
      [attemptId, req.user.id]
    );
    if (attempts.length === 0) {
      return res.status(404).json({ message: "Attempt not found" });
    }
    const attempt = attempts[0];
    // console.log("attempt", attempt);
    // console.log("attempt.correct_answers=>", attempt.correct_answers);
    // console.log(" attempt.total_questions=>", attempt.total_questions);
    const score = Math.round(
      (attempt.correct_answers / attempt.total_questions) * 100
    );
    await pool.execute(
      'UPDATE quiz_attempts SET status = "completed", score = ?, time_taken = ?, completed_at = NOW() WHERE id = ?',
      [score, timeTaken || 0, attemptId]
    );
    await pool.execute(
      "UPDATE users SET total_score = total_score + ?, quizzes_taken = quizzes_taken + 1 WHERE id = ?",
      [score, req.user.id]
    );
    res.json({
      score,
      correctAnswers: attempt.correct_answers,
      totalQuestions: attempt.total_questions,
    });
  } catch (error) {
    console.log("erroir", error);
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

exports.getQuizResult = async (req, res) => {
  try {
    const [attempts] = await pool.execute(
      `
            SELECT qa.*, q.title as quiz_title 
            FROM quiz_attempts qa 
            JOIN quizzes q ON qa.quiz_id = q.id 
            WHERE qa.id = ? AND qa.user_id = ?
        `,
      [req.params.attemptId, req.user.id]
    );
    if (attempts.length === 0) {
      return res.status(404).json({ message: "Result not found" });
    }
    res.json(attempts[0]);
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

exports.getAllScenarios = async (req, res) => {
  try {
    const [scenarios] = await pool.execute(
      "SELECT id, title, short_description, event_type, difficulty, category FROM scenarios WHERE is_active = TRUE"
    );
    res.json(scenarios);
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

exports.getScenarioById = async (req, res) => {
  try {
    const [scenarios] = await pool.execute(
      "SELECT * FROM scenarios WHERE id = ?",
      [req.params.id]
    );
    if (scenarios.length === 0) {
      return res.status(404).json({ message: "Scenario not found" });
    }
    res.json(scenarios[0]);
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};
