import { useState, useEffect, useRef } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import { scenariosAPI, progressAPI } from '../../services/api';
import { useAuth } from '../../context/ContextUsing';
import GivensTable from './GivensTable';
import AnswerOptions from './AnswerOptions';
import FeedbackModal from './FeedbackModal';
import Loading from '../common/Loading';
import Alert from '../common/Alert';
import { ArrowLeft, Clock, Target, BookOpen } from 'lucide-react';

const ScenarioDetail = () => {
    const { id } = useParams();
    const navigate = useNavigate();
    const { isAuthenticated } = useAuth();
    
    const [scenario, setScenario] = useState(null);
    const [answers, setAnswers] = useState([]);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState(null);
    const [selectedAnswer, setSelectedAnswer] = useState(null);
    const [submitting, setSubmitting] = useState(false);
    const [feedback, setFeedback] = useState(null);
    const [showFeedback, setShowFeedback] = useState(false);
    const [timeSpent, setTimeSpent] = useState(0);
    
    const timerRef = useRef(null);
    const startTimeRef = useRef(null);

    useEffect(() => {
        loadScenario();
        startTimeRef.current = Date.now();
        
        timerRef.current = setInterval(() => {
            setTimeSpent(Math.floor((Date.now() - startTimeRef.current) / 1000));
        }, 1000);

        return () => {
            if (timerRef.current) {
                clearInterval(timerRef.current);
            }
        };
    }, [id]);

    const loadScenario = async () => {
        try {
            setLoading(true);
            const response = await scenariosAPI.getById(id);
            setScenario(response.data.data.scenario);
            setAnswers(response.data.data.answers);
        } catch (err) {
            if (err.response?.status === 404) {
                setError('Scenario not found');
            } else {
                setError(err.response?.data?.message || 'Failed to load scenario');
            }
        } finally {
            setLoading(false);
        }
    };

    const handleAnswerSelect = (answer) => {
        if (showFeedback) return;
        setSelectedAnswer(answer.answer);
    };

    const handleSubmit = async () => {
        if (!selectedAnswer || !isAuthenticated) {
            if (!isAuthenticated) {
                navigate('/login', { state: { from: { pathname: `/scenarios/${id}` } } });
            }
            return;
        }

        setSubmitting(true);
        clearInterval(timerRef.current);

        try {
            const response = await progressAPI.submit({
                scenarioId: id,
                selectedAnswer,
                timeSpentSeconds: timeSpent
            });
            console.log("response.data.data=>",response.data.data)
            setFeedback(response.data.data.feedback);
            setShowFeedback(true);
        } catch (err) {
            setError(err.response?.data?.message || 'Failed to submit answer');
        } finally {
            setSubmitting(false);
        }
    };

    const formatTime = (seconds) => {
        const mins = Math.floor(seconds / 60);
        const secs = seconds % 60;
        return `${mins}:${secs.toString().padStart(2, '0')}`;
    };

    const difficultyColors = {
        easy: 'bg-green-100 text-green-800',
        medium: 'bg-yellow-100 text-yellow-800',
        hard: 'bg-orange-100 text-orange-800',
        expert: 'bg-red-100 text-red-800'
    };

    if (loading) {
        return (
            <div className="min-h-screen flex items-center justify-center">
                <Loading text="Loading scenario..." />
            </div>
        );
    }

    if (error) {
        return (
            <div className="max-w-4xl mx-auto px-4 py-8">
                <Alert type="error" message={error} />
                <button
                    onClick={() => navigate('/scenarios')}
                    className="mt-4 flex items-center text-indigo-600 hover:text-indigo-800"
                >
                    <ArrowLeft className="h-4 w-4 mr-2" />
                    Back to scenarios
                </button>
            </div>
        );
    }

    return (
        <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
            {/* Header */}
            <div className="mb-6">
                <button
                    onClick={() => navigate('/scenarios')}
                    className="flex items-center text-gray-600 hover:text-gray-900 mb-4"
                >
                    <ArrowLeft className="h-4 w-4 mr-2" />
                    Back to scenarios
                </button>

                <div className="flex items-start justify-between">
                    <div>
                        <div className="flex items-center space-x-3 mb-2">
                            {/* <span className="text-sm text-gray-500">{scenario.id}</span> */}
                            <span className={`px-2 py-1 text-xs font-medium rounded-full ${difficultyColors[scenario.difficultyLevel]}`}>
                                {scenario.difficultyLevel}
                            </span>
                            {scenario.category && (
                                <span className="px-2 py-1 text-xs font-medium rounded-full bg-indigo-100 text-indigo-800">
                                    {scenario.category}
                                </span>
                            )}
                        </div>
                        <h1 className="text-2xl font-bold text-gray-900">{scenario.title}</h1>
                    </div>
                    
                    <div className="flex items-center space-x-2 text-gray-600">
                        <Clock className="h-5 w-5" />
                        <span className="font-mono text-lg">{formatTime(timeSpent)}</span>
                    </div>
                </div>
            </div>

            {/* Scenario Content */}
            <div className="bg-white rounded-lg shadow-md p-6 mb-6">
                <div className="flex items-center space-x-2 mb-4">
                    <BookOpen className="h-5 w-5 text-indigo-600" />
                    <h2 className="text-lg font-semibold text-gray-900">Scenario</h2>
                </div>
                <p className="text-gray-700 leading-relaxed">{scenario.scenarioParagraph}</p>
            </div>

            {/* Givens Table */}
            <div className="bg-white rounded-lg shadow-md p-6 mb-6">
                <div className="flex items-center space-x-2 mb-4">
                    <Target className="h-5 w-5 text-indigo-600" />
                    <h2 className="text-lg font-semibold text-gray-900">Market Data</h2>
                </div>
                <GivensTable data={scenario.givensTable} />
            </div>

            {/* Answer Options */}
            <div className="bg-white rounded-lg shadow-md p-6 mb-6">
                <AnswerOptions
                    answers={answers}
                    onSelect={handleAnswerSelect}
                    disabled={showFeedback || submitting}
                    selectedAnswer={selectedAnswer}
                    correctAnswer={feedback?.correctAnswer || (feedback?.correct ? selectedAnswer : null)}
                    showFeedback={showFeedback}
                />

                {!showFeedback && (
                    <div className="mt-6 flex justify-end">
                        {!isAuthenticated && (
                            <p className="text-sm text-gray-500 mr-4 self-center">
                                Please <button onClick={() => navigate('/login')} className="text-indigo-600 hover:underline">login</button> to submit your answer
                            </p>
                        )}
                        <button
                            onClick={handleSubmit}
                            disabled={!selectedAnswer || submitting || !isAuthenticated}
                            className="px-6 py-2 bg-indigo-600 text-white font-medium rounded-lg hover:bg-indigo-700 disabled:opacity-50 disabled:cursor-not-allowed"
                        >
                            {submitting ? 'Submitting...' : 'Submit Answer'}
                        </button>
                    </div>
                )}
            </div>

            {/* Feedback Modal */}
            <FeedbackModal
                isOpen={showFeedback}
                onClose={() => setShowFeedback(false)}
                feedback={feedback}
                onNext={() => navigate('/scenarios')}
            />
        </div>
    );
};

export default ScenarioDetail;