import { useState, useEffect } from 'react';
import api from '../../services/api';
import Loading from '../Common/Loading';
import Modal from '../Common/Modal';
import { toast } from 'react-toastify';
import { FiPlus, FiEdit2, FiTrash2, FiSave } from 'react-icons/fi';

const emptyScenario = { id: '', title: '', short_description: '', givens_table: {}, scenario_paragraph: '', best_answer: '', best_answer_rationale: '', other_option1: '', other_option1_exp: '', other_option2: '', other_option2_exp: '', other_option3: '', other_option3_exp: '', event_type: 'normal', difficulty: 'medium', category: '' };

export default function ScenarioManager() {
  const [scenarios, setScenarios] = useState([]);
  const [loading, setLoading] = useState(true);
  const [editScenario, setEditScenario] = useState(null);
  const [showModal, setShowModal] = useState(false);
  const [isNew, setIsNew] = useState(false);

  const loadScenarios = () => {
    api.get('/admin/scenarios').then(res => setScenarios(res.data)).finally(() => setLoading(false));
  };

  useEffect(() => { loadScenarios(); }, []);

  const handleSubmit = async (e) => {
    e.preventDefault();
    try {
      if (isNew) {
        await api.post('/admin/scenarios', editScenario);
        toast.success('Scenario created successfully');
      } else {
        await api.put(`/admin/scenarios/${editScenario.id}`, editScenario);
        toast.success('Scenario updated successfully');
      }
      setShowModal(false);
      loadScenarios();
    } catch (error) {
      toast.error(error.response?.data?.message || 'Operation failed');
    }
  };

  const handleDelete = async (id) => {
    if (!confirm('Are you sure you want to delete this scenario?')) return;
    try {
      await api.delete(`/admin/scenarios/${id}`);
      toast.success('Scenario deleted successfully');
      loadScenarios();
    } catch (error) {
      toast.error(error.response?.data?.message || 'Delete failed');
    }
  };

  if (loading) return <Loading />;

  return (
    <div>
      <div className="flex justify-between items-center mb-8">
        <h1 className="text-3xl font-bold text-gray-800">Manage Scenarios</h1>
        <button onClick={() => { setEditScenario(emptyScenario); setIsNew(true); setShowModal(true); }} className="flex items-center gap-2 bg-indigo-600 text-white px-4 py-2 rounded-lg hover:bg-indigo-700">
          <FiPlus /> Add Scenario
        </button>
      </div>
      <div className="bg-white rounded-xl shadow-lg overflow-hidden">
        <table className="w-full">
          <thead className="bg-gray-50">
            <tr>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">ID</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Title</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Type</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Difficulty</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Actions</th>
            </tr>
          </thead>
          <tbody className="divide-y divide-gray-200">
            {scenarios.map(scenario => (
              <tr key={scenario.id}>
                <td className="px-6 py-4 text-gray-600 text-sm">{scenario.id}</td>
                <td className="px-6 py-4 font-medium text-gray-800">{scenario.title}</td>
                <td className="px-6 py-4">
                  <span className={`px-2 py-1 rounded text-sm ${scenario.event_type === 'major' ? 'bg-red-100 text-red-600' : 'bg-blue-100 text-blue-600'}`}>
                    {scenario.event_type}
                  </span>
                </td>
                <td className="px-6 py-4 text-gray-600 capitalize">{scenario.difficulty}</td>
                <td className="px-6 py-4">
                  <span className={`px-2 py-1 rounded text-sm ${scenario.is_active ? 'bg-green-100 text-green-600' : 'bg-gray-100 text-gray-600'}`}>
                    {scenario.is_active ? 'Active' : 'Inactive'}
                  </span>
                </td>
                <td className="px-6 py-4">
                  <button onClick={() => { setEditScenario(scenario); setIsNew(false); setShowModal(true); }} className="text-indigo-600 hover:text-indigo-800 mr-3"><FiEdit2 /></button>
                  <button onClick={() => handleDelete(scenario.id)} className="text-red-600 hover:text-red-800"><FiTrash2 /></button>
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
      <Modal isOpen={showModal} onClose={() => setShowModal(false)} title={isNew ? 'Add Scenario' : 'Edit Scenario'}>
        {editScenario && (
          <form onSubmit={handleSubmit} className="space-y-4 max-h-[70vh] overflow-y-auto">
            <div className="grid grid-cols-2 gap-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">ID</label>
                <input type="text" value={editScenario.id} onChange={(e) => setEditScenario({ ...editScenario, id: e.target.value })} className="w-full px-4 py-2 border border-gray-300 rounded-lg" required disabled={!isNew} />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Category</label>
                <input type="text" value={editScenario.category || ''} onChange={(e) => setEditScenario({ ...editScenario, category: e.target.value })} className="w-full px-4 py-2 border border-gray-300 rounded-lg" />
              </div>
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">Title</label>
              <input type="text" value={editScenario.title} onChange={(e) => setEditScenario({ ...editScenario, title: e.target.value })} className="w-full px-4 py-2 border border-gray-300 rounded-lg" required />
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">Short Description</label>
              <textarea value={editScenario.short_description || ''} onChange={(e) => setEditScenario({ ...editScenario, short_description: e.target.value })} className="w-full px-4 py-2 border border-gray-300 rounded-lg" rows="2" />
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">Scenario Paragraph</label>
              <textarea value={editScenario.scenario_paragraph} onChange={(e) => setEditScenario({ ...editScenario, scenario_paragraph: e.target.value })} className="w-full px-4 py-2 border border-gray-300 rounded-lg" rows="4" required />
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">Best Answer</label>
              <textarea value={editScenario.best_answer} onChange={(e) => setEditScenario({ ...editScenario, best_answer: e.target.value })} className="w-full px-4 py-2 border border-gray-300 rounded-lg" rows="2" required />
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">Best Answer Rationale</label>
              <textarea value={editScenario.best_answer_rationale || ''} onChange={(e) => setEditScenario({ ...editScenario, best_answer_rationale: e.target.value })} className="w-full px-4 py-2 border border-gray-300 rounded-lg" rows="2" />
            </div>
            <div className="grid grid-cols-2 gap-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Option 2</label>
                <textarea value={editScenario.other_option1 || ''} onChange={(e) => setEditScenario({ ...editScenario, other_option1: e.target.value })} className="w-full px-4 py-2 border border-gray-300 rounded-lg" rows="2" />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Option 2 Explanation</label>
                <textarea value={editScenario.other_option1_exp || ''} onChange={(e) => setEditScenario({ ...editScenario, other_option1_exp: e.target.value })} className="w-full px-4 py-2 border border-gray-300 rounded-lg" rows="2" />
              </div>
            </div>
            <div className="grid grid-cols-2 gap-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Option 3</label>
                <textarea value={editScenario.other_option2 || ''} onChange={(e) => setEditScenario({ ...editScenario, other_option2: e.target.value })} className="w-full px-4 py-2 border border-gray-300 rounded-lg" rows="2" />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Option 3 Explanation</label>
                <textarea value={editScenario.other_option2_exp || ''} onChange={(e) => setEditScenario({ ...editScenario, other_option2_exp: e.target.value })} className="w-full px-4 py-2 border border-gray-300 rounded-lg" rows="2" />
              </div>
            </div>
            <div className="grid grid-cols-2 gap-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Option 4</label>
                <textarea value={editScenario.other_option3 || ''} onChange={(e) => setEditScenario({ ...editScenario, other_option3: e.target.value })} className="w-full px-4 py-2 border border-gray-300 rounded-lg" rows="2" />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Option 4 Explanation</label>
                <textarea value={editScenario.other_option3_exp || ''} onChange={(e) => setEditScenario({ ...editScenario, other_option3_exp: e.target.value })} className="w-full px-4 py-2 border border-gray-300 rounded-lg" rows="2" />
              </div>
            </div>
            <div className="grid grid-cols-3 gap-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Event Type</label>
                <select value={editScenario.event_type} onChange={(e) => setEditScenario({ ...editScenario, event_type: e.target.value })} className="w-full px-4 py-2 border border-gray-300 rounded-lg">
                  <option value="normal">Normal</option>
                  <option value="major">Major</option>
                </select>
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Difficulty</label>
                <select value={editScenario.difficulty} onChange={(e) => setEditScenario({ ...editScenario, difficulty: e.target.value })} className="w-full px-4 py-2 border border-gray-300 rounded-lg">
                  <option value="easy">Easy</option>
                  <option value="medium">Medium</option>
                  <option value="hard">Hard</option>
                </select>
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Status</label>
                <select value={editScenario.is_active ? 'true' : 'false'} onChange={(e) => setEditScenario({ ...editScenario, is_active: e.target.value === 'true' })} className="w-full px-4 py-2 border border-gray-300 rounded-lg">
                  <option value="true">Active</option>
                  <option value="false">Inactive</option>
                </select>
              </div>
            </div>
            <button type="submit" className="flex items-center gap-2 bg-indigo-600 text-white px-4 py-2 rounded-lg hover:bg-indigo-700">
              <FiSave /> {isNew ? 'Create Scenario' : 'Save Changes'}
            </button>
          </form>
        )}
      </Modal>
    </div>
  );
}