import { useState, useEffect } from 'react';
import { Link } from 'react-router-dom';
import api from '../../services/api';
import Loading from '../Common/Loading';
import { FiClock, FiCheckCircle, FiXCircle } from 'react-icons/fi';

export default function History() {
  const [attempts, setAttempts] = useState([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    api.get('/user/history').then(res => setAttempts(res.data)).finally(() => setLoading(false));
  }, []);

  if (loading) return <Loading />;

  return (
    <div>
      <h1 className="text-3xl font-bold text-gray-800 mb-8">Quiz History</h1>
      {attempts.length === 0 ? (
        <div className="bg-white rounded-xl shadow-lg p-8 text-center">
          <p className="text-gray-500 text-lg">You haven't taken any quizzes yet.</p>
          <Link to="/" className="inline-block mt-4 bg-indigo-600 text-white px-6 py-2 rounded-lg hover:bg-indigo-700 transition">
            Browse Quizzes
          </Link>
        </div>
      ) : (
        <div className="bg-white rounded-xl shadow-lg overflow-hidden">
          <table className="w-full">
            <thead className="bg-gray-50">
              <tr>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Quiz</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Score</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Result</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Time</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Date</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Actions</th>
              </tr>
            </thead>
            <tbody className="divide-y divide-gray-200">
              {attempts.map(attempt => (
                <tr key={attempt.id}>
                  <td className="px-6 py-4 whitespace-nowrap font-medium text-gray-800">{attempt.quiz_title}</td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <span className={`font-semibold ${attempt.score >= 60 ? 'text-green-600' : 'text-red-600'}`}>
                      {attempt.score}%
                    </span>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    {attempt.score >= 60 ? (
                      <span className="flex items-center gap-1 text-green-600"><FiCheckCircle /> Passed</span>
                    ) : (
                      <span className="flex items-center gap-1 text-red-600"><FiXCircle /> Failed</span>
                    )}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-gray-500">
                    <span className="flex items-center gap-1"><FiClock /> {Math.floor(attempt.time_taken / 60)}:{String(attempt.time_taken % 60).padStart(2, '0')}</span>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-gray-500">
                    {new Date(attempt.started_at).toLocaleDateString()}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <Link to={`/result/${attempt.id}`} className="text-indigo-600 hover:underline">View</Link>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      )}
    </div>
  );
}