// src/models/Scenario.js
const db = require("../config/database");

class Scenario {
  static async findById(id) {
    const query = `
            SELECT * FROM scenarios 
            WHERE id = $1 AND is_active = true
        `;
    const result = await db.query(query, [id]);
    return result.rows[0];
  }

  static async findAll({ limit = 10, offset = 0, category, difficulty }) {
    let query = `
            SELECT id, title, short_description, difficulty_level, category, tags, created_at
            FROM scenarios 
            WHERE is_active = true
        `;
    const params = [];
    let paramCount = 0;

    if (category) {
      paramCount++;
      query += ` AND category = $${paramCount}`;
      params.push(category);
    }

    if (difficulty) {
      paramCount++;
      query += ` AND difficulty_level = $${paramCount}`;
      params.push(difficulty);
    }

    query += ` ORDER BY created_at DESC`;

    paramCount++;
    query += ` LIMIT $${paramCount}`;
    params.push(limit);

    paramCount++;
    query += ` OFFSET $${paramCount}`;
    params.push(offset);

    const result = await db.query(query, params);

    // Get total count
    let countQuery = "SELECT COUNT(*) FROM scenarios WHERE is_active = true";
    const countParams = [];
    let countParamNum = 0;

    if (category) {
      countParamNum++;
      countQuery += ` AND category = $${countParamNum}`;
      countParams.push(category);
    }
    if (difficulty) {
      countParamNum++;
      countQuery += ` AND difficulty_level = $${countParamNum}`;
      countParams.push(difficulty);
    }

    const countResult = await db.query(countQuery, countParams);

    return {
      scenarios: result.rows,
      total: parseInt(countResult.rows[0].count),
      limit,
      offset,
    };
  }

  static async getCategories() {
    const query = `
            SELECT DISTINCT category 
            FROM scenarios 
            WHERE is_active = true AND category IS NOT NULL
            ORDER BY category
        `;
    const result = await db.query(query);
    return result.rows.map((row) => row.category);
  }

  static async getStatistics(scenarioId) {
    const query = "SELECT * FROM scenario_statistics WHERE scenario_id = $1";
    const result = await db.query(query, [scenarioId]);
    return result.rows[0];
  }

  static async create(scenarioData) {
    const {
      title,
      short_description,
      givens_table,
      scenario_paragraph,
      best_answer,
      other_answers,
      difficulty_level,
      category,
      tags,
    } = scenarioData;

    const query = `
            INSERT INTO scenarios (
                title, short_description, givens_table, scenario_paragraph,
                best_answer, other_answers, difficulty_level, category, tags
            )
            VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9)
            RETURNING *
        `;

    const result = await db.query(query, [
      title,
      short_description,
      JSON.stringify(givens_table),
      scenario_paragraph,
      JSON.stringify(best_answer),
      JSON.stringify(other_answers),
      difficulty_level || "medium",
      category,
      tags,
    ]);

    return result.rows[0];
  }
}

module.exports = Scenario;
