import { useState, useEffect } from 'react';
import { scenariosAPI } from '../../services/api';
import ScenarioCard from './ScenarioCard';
import Loading from '../common/Loading';
import Alert from '../common/Alert';
import { Filter, Search } from 'lucide-react';

const ScenarioDashboard = () => {
    const [scenarios, setScenarios] = useState([]);
    const [categories, setCategories] = useState([]);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState(null);
    const [filters, setFilters] = useState({
        category: '',
        difficulty: '',
        search: ''
    });
    const [pagination, setPagination] = useState({
        limit: 12,
        offset: 0,
        total: 0
    });

    useEffect(() => {
        loadCategories();
    }, []);

    useEffect(() => {
        loadScenarios();
    }, [filters.category, filters.difficulty, pagination.offset]);

    const loadCategories = async () => {
        try {
            const response = await scenariosAPI.getCategories();
            setCategories(response.data.data);
        } catch (err) {
            console.error('Failed to load categories:', err);
        }
    };

    const loadScenarios = async () => {
        try {
            setLoading(true);
            const response = await scenariosAPI.getAll({
                limit: pagination.limit,
                offset: pagination.offset,
                category: filters.category || undefined,
                difficulty: filters.difficulty || undefined
            });
            setScenarios(response.data.data.scenarios);
            setPagination(prev => ({
                ...prev,
                total: response.data.data.total
            }));
        } catch (err) {
            setError(err.response?.data?.message || 'Failed to load scenarios');
        } finally {
            setLoading(false);
        }
    };

    const handleFilterChange = (key, value) => {
        setFilters(prev => ({ ...prev, [key]: value }));
        setPagination(prev => ({ ...prev, offset: 0 }));
    };

    const filteredScenarios = scenarios.filter(scenario =>
        scenario.title.toLowerCase().includes(filters.search.toLowerCase()) ||
        scenario.short_description.toLowerCase().includes(filters.search.toLowerCase())
    );

    const totalPages = Math.ceil(pagination.total / pagination.limit);
    const currentPage = Math.floor(pagination.offset / pagination.limit) + 1;

    return (
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
            <div className="mb-8">
                <h1 className="text-3xl font-bold text-gray-900">Scenarios</h1>
                <p className="mt-2 text-gray-600">
                    Practice your decision-making skills with real-world scenarios
                </p>
            </div>

            {/* Filters */}
            <div className="mb-6 flex flex-col sm:flex-row gap-4">
                <div className="flex-1 relative">
                    <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-5 w-5 text-gray-400" />
                    <input
                        type="text"
                        placeholder="Search scenarios..."
                        value={filters.search}
                        onChange={(e) => handleFilterChange('search', e.target.value)}
                        className="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent"
                    />
                </div>
                
                <div className="flex gap-4">
                    <select
                        value={filters.category}
                        onChange={(e) => handleFilterChange('category', e.target.value)}
                        className="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent"
                    >
                        <option value="">All Categories</option>
                        {categories.map(cat => (
                            <option key={cat} value={cat}>{cat}</option>
                        ))}
                    </select>

                    <select
                        value={filters.difficulty}
                        onChange={(e) => handleFilterChange('difficulty', e.target.value)}
                        className="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent"
                    >
                        <option value="">All Difficulties</option>
                        <option value="easy">Easy</option>
                        <option value="medium">Medium</option>
                        <option value="hard">Hard</option>
                        <option value="expert">Expert</option>
                    </select>
                </div>
            </div>

            {error && <Alert type="error" message={error} onClose={() => setError(null)} />}

            {loading ? (
                <Loading text="Loading scenarios..." />
            ) : filteredScenarios.length === 0 ? (
                <div className="text-center py-12">
                    <Filter className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                    <h3 className="text-lg font-medium text-gray-900">No scenarios found</h3>
                    <p className="text-gray-600">Try adjusting your filters</p>
                </div>
            ) : (
                <>
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                        {filteredScenarios.map(scenario => (
                            <ScenarioCard key={scenario.id} scenario={scenario} />
                        ))}
                    </div>

                    {/* Pagination */}
                    {totalPages > 1 && (
                        <div className="mt-8 flex justify-center space-x-2">
                            <button
                                onClick={() => setPagination(prev => ({
                                    ...prev,
                                    offset: Math.max(0, prev.offset - prev.limit)
                                }))}
                                disabled={currentPage === 1}
                                className="px-4 py-2 border border-gray-300 rounded-md disabled:opacity-50 disabled:cursor-not-allowed hover:bg-gray-50"
                            >
                                Previous
                            </button>
                            <span className="px-4 py-2 text-gray-600">
                                Page {currentPage} of {totalPages}
                            </span>
                            <button
                                onClick={() => setPagination(prev => ({
                                    ...prev,
                                    offset: prev.offset + prev.limit
                                }))}
                                disabled={currentPage === totalPages}
                                className="px-4 py-2 border border-gray-300 rounded-md disabled:opacity-50 disabled:cursor-not-allowed hover:bg-gray-50"
                            >
                                Next
                            </button>
                        </div>
                    )}
                </>
            )}
        </div>
    );
};

export default ScenarioDashboard;