import { useState, useEffect } from 'react';
// import { useAuth } from '../../context/AuthContext';
import { authAPI, progressAPI } from '../../services/api';
import Loading from '../common/Loading';
import { User, Mail, Calendar, Award, Target, Clock, TrendingUp } from 'lucide-react';

const UserProfile = () => {
    // const { user } = useAuth();
    const [profile, setProfile] = useState(null);
    const [progress, setProgress] = useState([]);
    const [loading, setLoading] = useState(true);

    useEffect(() => {
        console.log("loadProfileData");
        loadProfileData();
    }, []);

    const loadProfileData = async () => {
        try {
            const [profileRes, progressRes] = await Promise.all([
                authAPI.getProfile(),
                progressAPI.getUserProgress({ limit: 10 })
            ]);
            setProfile(profileRes.data.data);
            setProgress(progressRes.data.data);
        } catch (err) {
            console.error('Failed to load profile:', err);
        } finally {
            setLoading(false);
        }
    };

    if (loading) {
        return (
            <div className="min-h-screen flex items-center justify-center">
                <Loading text="Loading profile..." />
            </div>
        );
    }

    const stats = profile?.statistics || {};

    return (
        <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
            {/* Profile Header */}
            <div className="bg-white rounded-lg shadow-md p-6 mb-6">
                <div className="flex items-center space-x-4">
                    <div className="h-20 w-20 rounded-full bg-indigo-100 flex items-center justify-center">
                        <User className="h-10 w-10 text-indigo-600" />
                    </div>
                    <div>
                        <h1 className="text-2xl font-bold text-gray-900">{profile?.user?.username}</h1>
                        <div className="flex items-center text-gray-600 mt-1">
                            <Mail className="h-4 w-4 mr-2" />
                            {profile?.user?.email}
                        </div>
                        <div className="flex items-center text-gray-500 text-sm mt-1">
                            <Calendar className="h-4 w-4 mr-2" />
                            Joined {new Date(profile?.user?.created_at).toLocaleDateString()}
                        </div>
                    </div>
                </div>
            </div>

            {/* Statistics */}
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
                <div className="bg-white rounded-lg shadow-md p-4">
                    <div className="flex items-center justify-between">
                        <div>
                            <p className="text-sm text-gray-500">Scenarios</p>
                            <p className="text-2xl font-bold text-gray-900">{stats.scenarios_attempted || 0}</p>
                        </div>
                        <Target className="h-8 w-8 text-indigo-600" />
                    </div>
                </div>

                <div className="bg-white rounded-lg shadow-md p-4">
                    <div className="flex items-center justify-between">
                        <div>
                            <p className="text-sm text-gray-500">Correct</p>
                            <p className="text-2xl font-bold text-green-600">{stats.correct_answers || 0}</p>
                        </div>
                        <Award className="h-8 w-8 text-green-600" />
                    </div>
                </div>

                <div className="bg-white rounded-lg shadow-md p-4">
                    <div className="flex items-center justify-between">
                        <div>
                            <p className="text-sm text-gray-500">Avg Score</p>
                            <p className="text-2xl font-bold text-gray-900">{stats.average_score || 0}</p>
                        </div>
                        <TrendingUp className="h-8 w-8 text-blue-600" />
                    </div>
                </div>

                <div className="bg-white rounded-lg shadow-md p-4">
                    <div className="flex items-center justify-between">
                        <div>
                            <p className="text-sm text-gray-500">Accuracy</p>
                            <p className="text-2xl font-bold text-purple-600">{stats.accuracy_percentage || 0}%</p>
                        </div>
                        <Clock className="h-8 w-8 text-purple-600" />
                    </div>
                </div>
            </div>

            {/* Recent Progress */}
            <div className="bg-white rounded-lg shadow-md p-6">
                <h2 className="text-lg font-semibold text-gray-900 mb-4">Recent Activity</h2>
                
                {progress.length === 0 ? (
                    <p className="text-gray-500 text-center py-8">
                        No activity yet. Start practicing scenarios!
                    </p>
                ) : (
                    <div className="space-y-4">
                        {progress.map((item) => (
                            <div
                                key={item.id}
                                className="flex items-center justify-between p-4 bg-gray-50 rounded-lg"
                            >
                                <div>
                                    <h3 className="font-medium text-gray-900">{item.scenario_title}</h3>
                                    <p className="text-sm text-gray-500">
                                        {new Date(item.completed_at).toLocaleDateString()} • Attempt #{item.attempt_number}
                                    </p>
                                </div>
                                <div className="flex items-center space-x-4">
                                    <span className={`px-2 py-1 text-xs font-medium rounded-full ${
                                        item.is_correct ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'
                                    }`}>
                                        {item.is_correct ? 'Correct' : 'Incorrect'}
                                    </span>
                                    <span className="text-lg font-bold text-gray-900">{item.score}</span>
                                </div>
                            </div>
                        ))}
                    </div>
                )}
            </div>
        </div>
    );
};

export default UserProfile;