// src/middleware/validation.js
const { body, param, query, validationResult } = require("express-validator");

const handleValidationErrors = (req, res, next) => {
  const errors = validationResult(req);
  if (!errors.isEmpty()) {
    return res.status(400).json({
      success: false,
      message: "Validation failed",
      errors: errors.array().map((err) => ({
        field: err.path,
        message: err.msg,
      })),
    });
  }
  next();
};

const registerValidation = [
  body("username")
    .trim()
    .isLength({ min: 3, max: 50 })
    .withMessage("Username must be between 3 and 50 characters")
    .matches(/^[a-zA-Z0-9_]+$/)
    .withMessage("Username can only contain letters, numbers, and underscores"),
  body("email")
    .trim()
    .isEmail()
    .withMessage("Invalid email address")
    .normalizeEmail(),
  body("password")
    .isLength({ min: 8 })
    .withMessage("Password must be at least 8 characters")
    .matches(/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)/)
    .withMessage(
      "Password must contain at least one lowercase letter, one uppercase letter, and one number"
    ),
  handleValidationErrors,
];

const loginValidation = [
  body("email")
    .trim()
    .isEmail()
    .withMessage("Invalid email address")
    .normalizeEmail(),
  body("password").notEmpty().withMessage("Password is required"),
  handleValidationErrors,
];

const scenarioIdValidation = [
  param("id")
    .matches(/^SCEN-\d{6}$/)
    .withMessage("Invalid scenario ID format"),
  handleValidationErrors,
];

const progressValidation = [
  body("scenarioId")
    .matches(/^SCEN-\d{6}$/)
    .withMessage("Invalid scenario ID format"),
  body("selectedAnswer")
    .trim()
    .notEmpty()
    .withMessage("Selected answer is required"),
  body("timeSpentSeconds")
    .optional()
    .isInt({ min: 0 })
    .withMessage("Time spent must be a positive integer"),
  handleValidationErrors,
];

const paginationValidation = [
  query("limit")
    .optional()
    .isInt({ min: 1, max: 100 })
    .withMessage("Limit must be between 1 and 100"),
  query("offset")
    .optional()
    .isInt({ min: 0 })
    .withMessage("Offset must be a non-negative integer"),
  handleValidationErrors,
];

module.exports = {
  registerValidation,
  loginValidation,
  scenarioIdValidation,
  progressValidation,
  paginationValidation,
  handleValidationErrors,
};
