const pool = require("../config/database");

exports.getDashboard = async (req, res) => {
  try {
    const [userCount] = await pool.execute(
      "SELECT COUNT(*) as count FROM users"
    );
    const [quizCount] = await pool.execute(
      "SELECT COUNT(*) as count FROM quizzes"
    );
    const [scenarioCount] = await pool.execute(
      "SELECT COUNT(*) as count FROM scenarios"
    );
    const [attemptCount] = await pool.execute(
      'SELECT COUNT(*) as count FROM quiz_attempts WHERE status = "completed"'
    );
    const [recentAttempts] = await pool.execute(`
            SELECT qa.*, u.username, q.title as quiz_title 
            FROM quiz_attempts qa 
            JOIN users u ON qa.user_id = u.id 
            JOIN quizzes q ON qa.quiz_id = q.id 
            ORDER BY qa.completed_at DESC LIMIT 10
        `);
    res.json({
      stats: {
        users: userCount[0].count,
        quizzes: quizCount[0].count,
        scenarios: scenarioCount[0].count,
        attempts: attemptCount[0].count,
      },
      recentAttempts,
    });
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

exports.getAllUsers = async (req, res) => {
  try {
    const [users] = await pool.execute(
      "SELECT id, username, email, role, total_score, quizzes_taken, created_at FROM users ORDER BY created_at DESC"
    );
    res.json(users);
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

exports.updateUser = async (req, res) => {
  try {
    const { username, email, role } = req.body;
    await pool.execute(
      "UPDATE users SET username = ?, email = ?, role = ? WHERE id = ?",
      [username, email, role, req.params.id]
    );
    res.json({ message: "User updated successfully" });
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

exports.deleteUser = async (req, res) => {
  try {
    await pool.execute("DELETE FROM users WHERE id = ?", [req.params.id]);
    res.json({ message: "User deleted successfully" });
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

exports.getAllScenarios = async (req, res) => {
  try {
    const [scenarios] = await pool.execute(
      "SELECT * FROM scenarios ORDER BY created_at DESC"
    );
    res.json(scenarios);
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

exports.createScenario = async (req, res) => {
  try {
    const {
      id,
      title,
      short_description,
      givens_table,
      scenario_paragraph,
      best_answer,
      best_answer_rationale,
      other_option1,
      other_option1_exp,
      other_option2,
      other_option2_exp,
      other_option3,
      other_option3_exp,
      event_type,
      difficulty,
      category,
    } = req.body;
    await pool.execute(
      `
            INSERT INTO scenarios (id, title, short_description, givens_table, scenario_paragraph, best_answer, best_answer_rationale, other_option1, other_option1_exp, other_option2, other_option2_exp, other_option3, other_option3_exp, event_type, difficulty, category)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        `,
      [
        id,
        title,
        short_description,
        JSON.stringify(givens_table),
        scenario_paragraph,
        best_answer,
        best_answer_rationale,
        other_option1,
        other_option1_exp,
        other_option2,
        other_option2_exp,
        other_option3,
        other_option3_exp,
        event_type,
        difficulty,
        category,
      ]
    );
    res.status(201).json({ message: "Scenario created successfully" });
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

exports.updateScenario = async (req, res) => {
  try {
    const {
      title,
      short_description,
      givens_table,
      scenario_paragraph,
      best_answer,
      best_answer_rationale,
      other_option1,
      other_option1_exp,
      other_option2,
      other_option2_exp,
      other_option3,
      other_option3_exp,
      event_type,
      difficulty,
      category,
      is_active,
    } = req.body;
    await pool.execute(
      `
            UPDATE scenarios SET title = ?, short_description = ?, givens_table = ?, scenario_paragraph = ?, best_answer = ?, best_answer_rationale = ?, other_option1 = ?, other_option1_exp = ?, other_option2 = ?, other_option2_exp = ?, other_option3 = ?, other_option3_exp = ?, event_type = ?, difficulty = ?, category = ?, is_active = ?
            WHERE id = ?
        `,
      [
        title,
        short_description,
        JSON.stringify(givens_table),
        scenario_paragraph,
        best_answer,
        best_answer_rationale,
        other_option1,
        other_option1_exp,
        other_option2,
        other_option2_exp,
        other_option3,
        other_option3_exp,
        event_type,
        difficulty,
        category,
        is_active,
        req.params.id,
      ]
    );
    res.json({ message: "Scenario updated successfully" });
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

exports.deleteScenario = async (req, res) => {
  try {
    await pool.execute("DELETE FROM scenarios WHERE id = ?", [req.params.id]);
    res.json({ message: "Scenario deleted successfully" });
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

exports.getAllQuizzes = async (req, res) => {
  try {
    const [quizzes] = await pool.execute(`
            SELECT q.*, u.username as creator_name,
            (SELECT COUNT(*) FROM quiz_scenarios WHERE quiz_id = q.id) as question_count
            FROM quizzes q
            LEFT JOIN users u ON q.created_by = u.id
            ORDER BY q.created_at DESC
        `);
    res.json(quizzes);
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

exports.createQuiz = async (req, res) => {
  try {
    const { title, description, time_limit, passing_score, scenario_ids } =
      req.body;
    const [result] = await pool.execute(
      "INSERT INTO quizzes (title, description, time_limit, passing_score, created_by) VALUES (?, ?, ?, ?, ?)",
      [title, description, time_limit, passing_score, req.user.id]
    );
    const quizId = result.insertId;
    for (let i = 0; i < scenario_ids.length; i++) {
      await pool.execute(
        "INSERT INTO quiz_scenarios (quiz_id, scenario_id, question_order) VALUES (?, ?, ?)",
        [quizId, scenario_ids[i], i + 1]
      );
    }
    res.status(201).json({ message: "Quiz created successfully", quizId });
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

exports.updateQuiz = async (req, res) => {
  try {
    const {
      title,
      description,
      time_limit,
      passing_score,
      is_active,
      scenario_ids,
    } = req.body;
    await pool.execute(
      "UPDATE quizzes SET title = ?, description = ?, time_limit = ?, passing_score = ?, is_active = ? WHERE id = ?",
      [title, description, time_limit, passing_score, is_active, req.params.id]
    );
    if (scenario_ids) {
      await pool.execute("DELETE FROM quiz_scenarios WHERE quiz_id = ?", [
        req.params.id,
      ]);
      for (let i = 0; i < scenario_ids.length; i++) {
        await pool.execute(
          "INSERT INTO quiz_scenarios (quiz_id, scenario_id, question_order) VALUES (?, ?, ?)",
          [req.params.id, scenario_ids[i], i + 1]
        );
      }
    }
    res.json({ message: "Quiz updated successfully" });
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

exports.deleteQuiz = async (req, res) => {
  try {
    await pool.execute("DELETE FROM quizzes WHERE id = ?", [req.params.id]);
    res.json({ message: "Quiz deleted successfully" });
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};
