import { useState, useEffect } from 'react';
import { Link } from 'react-router-dom';
import api from '../../services/api';
import Loading from '../Common/Loading';
import { FiUsers, FiFileText, FiList, FiCheckCircle } from 'react-icons/fi';

export default function Dashboard() {
  const [data, setData] = useState(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    api.get('/admin/dashboard').then(res => setData(res.data)).finally(() => setLoading(false));
  }, []);

  if (loading) return <Loading />;

  const stats = [
    { label: 'Total Users', value: data.stats.users, icon: FiUsers, color: 'bg-blue-500', link: '/admin/users' },
    { label: 'Quizzes', value: data.stats.quizzes, icon: FiList, color: 'bg-green-500', link: '/admin/quizzes' },
    { label: 'Scenarios', value: data.stats.scenarios, icon: FiFileText, color: 'bg-purple-500', link: '/admin/scenarios' },
    { label: 'Attempts', value: data.stats.attempts, icon: FiCheckCircle, color: 'bg-orange-500', link: '#' }
  ];

  return (
    <div>
      <h1 className="text-3xl font-bold text-gray-800 mb-8">Admin Dashboard</h1>
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
        {stats.map((stat, index) => (
          <Link key={index} to={stat.link} className="bg-white rounded-xl shadow-lg p-6 card-hover">
            <div className={`w-12 h-12 ${stat.color} rounded-lg flex items-center justify-center mb-4`}>
              <stat.icon className="text-white" size={24} />
            </div>
            <div className="text-3xl font-bold text-gray-800">{stat.value}</div>
            <div className="text-gray-500">{stat.label}</div>
          </Link>
        ))}
      </div>
      <div className="bg-white rounded-xl shadow-lg p-6">
        <h2 className="text-xl font-semibold text-gray-800 mb-4">Recent Activity</h2>
        {data.recentAttempts.length === 0 ? (
          <p className="text-gray-500">No recent activity</p>
        ) : (
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead className="bg-gray-50">
                <tr>
                  <th className="px-4 py-2 text-left text-xs font-medium text-gray-500 uppercase">User</th>
                  <th className="px-4 py-2 text-left text-xs font-medium text-gray-500 uppercase">Quiz</th>
                  <th className="px-4 py-2 text-left text-xs font-medium text-gray-500 uppercase">Score</th>
                  <th className="px-4 py-2 text-left text-xs font-medium text-gray-500 uppercase">Date</th>
                </tr>
              </thead>
              <tbody className="divide-y divide-gray-200">
                {data.recentAttempts.map(attempt => (
                  <tr key={attempt.id}>
                    <td className="px-4 py-2 text-gray-800">{attempt.username}</td>
                    <td className="px-4 py-2 text-gray-600">{attempt.quiz_title}</td>
                    <td className="px-4 py-2">
                      <span className={attempt.score >= 60 ? 'text-green-600' : 'text-red-600'}>{attempt.score}%</span>
                    </td>
                    <td className="px-4 py-2 text-gray-500">{attempt.completed_at ? new Date(attempt.completed_at).toLocaleString() : '-'}</td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}
      </div>
    </div>
  );
}