const bcrypt = require("bcryptjs");
const pool = require("../config/database");

exports.getProfile = async (req, res) => {
  try {
    const [users] = await pool.execute(
      "SELECT id, username, email, role, total_score, quizzes_taken, avatar, created_at FROM users WHERE id = ?",
      [req.user.id]
    );
    res.json(users[0]);
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

exports.updateProfile = async (req, res) => {
  try {
    const { username, email } = req.body;
    await pool.execute(
      "UPDATE users SET username = ?, email = ? WHERE id = ?",
      [username, email, req.user.id]
    );
    res.json({ message: "Profile updated successfully" });
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

exports.changePassword = async (req, res) => {
  try {
    const { currentPassword, newPassword } = req.body;
    const [users] = await pool.execute(
      "SELECT password FROM users WHERE id = ?",
      [req.user.id]
    );
    const isMatch = await bcrypt.compare(currentPassword, users[0].password);
    if (!isMatch) {
      return res.status(400).json({ message: "Current password is incorrect" });
    }
    const salt = await bcrypt.genSalt(10);
    const hashedPassword = await bcrypt.hash(newPassword, salt);
    await pool.execute("UPDATE users SET password = ? WHERE id = ?", [
      hashedPassword,
      req.user.id,
    ]);
    res.json({ message: "Password changed successfully" });
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

exports.getHistory = async (req, res) => {
  try {
    const [attempts] = await pool.execute(
      `
            SELECT qa.*, q.title as quiz_title
            FROM quiz_attempts qa
            JOIN quizzes q ON qa.quiz_id = q.id
            WHERE qa.user_id = ? 
            AND qa.total_questions > 0 
            ORDER BY qa.started_at DESC
        `,
      [req.user.id]
    );
    res.json(attempts);
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

exports.getLeaderboard = async (req, res) => {
  try {
    const [leaderboard] = await pool.execute(`
            SELECT id, username, total_score, quizzes_taken, avatar
            FROM users
            WHERE quizzes_taken > 0
            ORDER BY total_score DESC, quizzes_taken DESC
            LIMIT 50
        `);
    res.json(leaderboard);
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};
