// src/models/Progress.js
const db = require("../config/database");

class Progress {
  static async create({
    userId,
    scenarioId,
    selectedAnswer,
    isCorrect,
    score,
    timeSpentSeconds,
  }) {
    const query = `
            INSERT INTO user_progress (
                user_id, scenario_id, selected_answer, is_correct, score, time_spent_seconds
            )
            VALUES ($1, $2, $3, $4, $5, $6)
            RETURNING *
        `;

    const result = await db.query(query, [
      userId,
      scenarioId,
      selectedAnswer,
      isCorrect,
      score,
      timeSpentSeconds,
    ]);

    return result.rows[0];
  }

  static async findByUserAndScenario(userId, scenarioId) {
    const query = `
            SELECT * FROM user_progress 
            WHERE user_id = $1 AND scenario_id = $2
            ORDER BY attempt_number DESC
        `;
    const result = await db.query(query, [userId, scenarioId]);
    return result.rows;
  }

  static async getUserProgress(userId, { limit = 10, offset = 0 }) {
    const query = `
            SELECT 
                up.*,
                s.title as scenario_title,
                s.difficulty_level,
                s.category
            FROM user_progress up
            JOIN scenarios s ON up.scenario_id = s.id
            WHERE up.user_id = $1
            ORDER BY up.completed_at DESC
            LIMIT $2 OFFSET $3
        `;
    const result = await db.query(query, [userId, limit, offset]);
    return result.rows;
  }

  static async getCompletedScenarioIds(userId) {
    const query = `
            SELECT DISTINCT scenario_id 
            FROM user_progress 
            WHERE user_id = $1 AND is_correct = true
        `;
    const result = await db.query(query, [userId]);
    return result.rows.map((row) => row.scenario_id);
  }

  static async getLeaderboard(limit = 10) {
    const query = `
            SELECT 
                u.username,
                COUNT(DISTINCT up.scenario_id) as scenarios_completed,
                ROUND(AVG(up.score)::NUMERIC, 2) as average_score,
                SUM(CASE WHEN up.is_correct THEN 1 ELSE 0 END) as correct_answers
            FROM users u
            JOIN user_progress up ON u.id = up.user_id
            GROUP BY u.id, u.username
            ORDER BY average_score DESC, scenarios_completed DESC
            LIMIT $1
        `;
    const result = await db.query(query, [limit]);
    return result.rows;
  }
}

module.exports = Progress;
