// src/controllers/authController.js
const jwt = require("jsonwebtoken");
const crypto = require("crypto");
const User = require("../models/User");
const jwtConfig = require("../config/jwt");

const generateTokens = (user) => {
  const accessToken = jwt.sign(
    { userId: user.id, email: user.email, username: user.username },
    jwtConfig.accessToken.secret,
    { expiresIn: jwtConfig.accessToken.expiry }
  );

  const refreshToken = jwt.sign(
    { userId: user.id, tokenId: crypto.randomUUID() },
    jwtConfig.refreshToken.secret,
    { expiresIn: jwtConfig.refreshToken.expiry }
  );

  return { accessToken, refreshToken };
};

const register = async (req, res, next) => {
  try {
    const { username, email, password } = req.body;

    // Check if user already exists
    const existingEmail = await User.findByEmail(email);
    if (existingEmail) {
      return res.status(409).json({
        success: false,
        message: "Email already registered",
      });
    }

    const existingUsername = await User.findByUsername(username);
    if (existingUsername) {
      return res.status(409).json({
        success: false,
        message: "Username already taken",
      });
    }

    // Create user
    const user = await User.create({ username, email, password });

    // Generate tokens
    const { accessToken, refreshToken } = generateTokens(user);

    // Hash and store refresh token
    const refreshTokenHash = crypto
      .createHash("sha256")
      .update(refreshToken)
      .digest("hex");

    const expiresAt = new Date(Date.now() + 7 * 24 * 60 * 60 * 1000); // 7 days
    await User.saveRefreshToken(
      user.id,
      refreshTokenHash,
      expiresAt,
      req.ip,
      req.headers["user-agent"]
    );

    res.status(201).json({
      success: true,
      message: "Registration successful",
      data: {
        user: {
          id: user.id,
          username: user.username,
          email: user.email,
        },
        accessToken,
        refreshToken,
      },
    });
  } catch (error) {
    next(error);
  }
};

const login = async (req, res, next) => {
  try {
    const { email, password } = req.body;

    // Find user
    const user = await User.findByEmail(email);
    if (!user) {
      return res.status(401).json({
        success: false,
        message: "Invalid email or password",
      });
    }

    // Verify password
    const isValidPassword = await User.verifyPassword(
      password,
      user.password_hash
    );
    if (!isValidPassword) {
      return res.status(401).json({
        success: false,
        message: "Invalid email or password",
      });
    }

    // Generate tokens
    const { accessToken, refreshToken } = generateTokens(user);

    // Hash and store refresh token
    const refreshTokenHash = crypto
      .createHash("sha256")
      .update(refreshToken)
      .digest("hex");
    console.log("refreshToken=>", refreshToken);
    console.log("refreshTokenHash=>", refreshTokenHash);
    const expiresAt = new Date(Date.now() + 7 * 24 * 60 * 60 * 1000);
    const r = await User.saveRefreshToken(
      user.id,
      refreshTokenHash,
      expiresAt,
      req.ip,
      req.headers["user-agent"]
    );
    console.log("RR=>", r);
    res.json({
      success: true,
      message: "Login successful",
      data: {
        user: {
          id: user.id,
          username: user.username,
          email: user.email,
        },
        accessToken,
        refreshToken,
      },
    });
  } catch (error) {
    next(error);
  }
};

const refreshToken = async (req, res, next) => {
  try {
    const { refreshToken: token } = req.body;

    if (!token) {
      return res.status(400).json({
        success: false,
        message: "Refresh token required",
      });
    }

    // Verify token
    let decoded;
    try {
      decoded = jwt.verify(token, jwtConfig.refreshToken.secret);
    } catch (err) {
      return res.status(401).json({
        success: false,
        message: "Invalid refresh token",
      });
    }

    // Check if token exists in database
    const tokenHash = crypto.createHash("sha256").update(token).digest("hex");
    console.log("Token=>>", token);
    console.log("tokenHash=>>", tokenHash);
    const storedToken = await User.findRefreshToken(tokenHash);
    console.log("storedToken", storedToken);
    if (!storedToken) {
      return res.status(401).json({
        success: false,
        message: "Refresh token not found or expired",
      });
    }

    // Get user
    const user = await User.findById(decoded.userId);
    if (!user) {
      return res.status(401).json({
        success: false,
        message: "User not found",
      });
    }

    const tokens = generateTokens(user);

    const newTokenHash = crypto
      .createHash("sha256")
      .update(tokens.refreshToken)
      .digest("hex");

    const expiresAt = new Date(Date.now() + 7 * 24 * 60 * 60 * 1000);

    await User.revokeRefreshToken(tokenHash);
    await User.saveRefreshToken(
      user.id,
      newTokenHash,
      expiresAt,
      req.ip,
      req.headers["user-agent"]
    );

    res.json({
      success: true,
      data: {
        accessToken: tokens.accessToken,
        refreshToken: tokens.refreshToken,
      },
    });
  } catch (error) {
    next(error);
  }
};

const logout = async (req, res, next) => {
  try {
    const { refreshToken: token } = req.body;

    if (token) {
      const tokenHash = crypto.createHash("sha256").update(token).digest("hex");
      await User.revokeRefreshToken(tokenHash);
    }

    res.json({
      success: true,
      message: "Logged out successfully",
    });
  } catch (error) {
    next(error);
  }
};

const getProfile = async (req, res, next) => {
  try {
    const user = await User.findById(req.user.userId);
    if (!user) {
      return res.status(404).json({
        success: false,
        message: "User not found",
      });
    }

    const statistics = await User.getStatistics(req.user.userId);

    res.json({
      success: true,
      data: {
        user,
        statistics,
      },
    });
  } catch (error) {
    next(error);
  }
};

module.exports = {
  register,
  login,
  refreshToken,
  logout,
  getProfile,
};
