from abc import ABC, abstractmethod
from typing import List, Dict

class BaseTTSService(ABC):
    """
    Abstract Base Class (the "Contract") for all Text-to-Speech services.
    It ensures that any TTS service we create has a consistent interface.
    """

    @abstractmethod
    def is_available(self) -> bool:
        """Check if the TTS service is configured and available."""
        pass

    @abstractmethod
    def generate_speech(self, text: str, language: str = "en") -> bytes:
        """
        Generate speech from text.

        Args:
            text (str): The text to synthesize.
            language (str): The language of the text (e.g., "en", "ar").

        Returns:
            bytes: The raw audio data of the speech (e.g., in WAV or MP3 format).
        """
        pass
    
    @abstractmethod
    def generate_speech_from_sequence(self, segments: List[Dict[str, str]]) -> bytes:
        """
        Generates a single audio file from a list of language-tagged text segments.
        This is for handling mixed-language sentences.
        """
        pass