import requests
import json

# The base URL of your locally running FastAPI application
BASE_URL = "https://voice-agent.caprover.al-arcade.com"

# --- Test Configuration ---
# Use a student ID that has some recent chat history on any topic.
TEST_STUDENT_ID = "student_001" 
# Change this to a normal phrase or the trigger phrase to test different paths
TRIGGER_PHRASE = "اسألني سؤال" # "Ask me a question"
# TRIGGER_PHRASE = "ما هو التكيف؟" # "What is adaptation?"

def test_chat_endpoint(student_id: str, message: str):
    """
    Sends a single request to the /chat endpoint and prints the full response.
    """
    endpoint = f"{BASE_URL}/chat"
    
    payload = {
        "student_id": student_id,
        "text": message,
    }

    print(f"▶️  Sending message to /chat for student '{student_id}'...")
    print(f"    Message: '{message}'")
    
    try:
        # Make the POST request
        response = requests.post(endpoint, data=payload, timeout=120)
        
        # Print the HTTP status code and headers for context
        print(f"\n--- API Response from /chat ---")
        print(f"Status Code: {response.status_code}")
        print("Headers:")
        for key, value in response.headers.items():
            print(f"  {key}: {value}")
        
        # Try to parse and pretty-print the JSON response body
        try:
            response_data = response.json()
            print("\nResponse Body (JSON):")
            print(json.dumps(response_data, indent=2, ensure_ascii=False))
        except json.JSONDecodeError:
            print("\nResponse Body (Not JSON):")
            print(response.text)

    except requests.exceptions.RequestException as e:
        print(f"\n❌ FAILED: An error occurred while making the request: {e}")


if __name__ == "__main__":
    print("="*60)
    print("  SIMPLE /chat ENDPOINT TEST")
    print("  This script sends one message and prints the immediate response.")
    print("="*60)
    
    test_chat_endpoint(TEST_STUDENT_ID, TRIGGER_PHRASE)
    
    print("\n" + "="*60)
    print("  Test complete.")
    print("="*60)