# --- START OF FILE schemas/agent.py ---
from pydantic import BaseModel, Field, validator
from typing import Optional, List, Dict, Union
from enum import Enum
import os
from pathlib import Path
import sys
sys.path.append(os.path.abspath(os.path.join(os.path.dirname(__file__), '..')))
from core import StudentNationality, StudyLanguage

# --- Student Context Models ---

class StudentProfile(BaseModel):
    student_id: str
    student_name: str
    grade: int = Field(..., ge=1, le=12)  # Validate grade is between 1 and 12
    study_language: StudyLanguage
    nationality: Union[StudentNationality, str] # Handle both Enum obj and string from DB
    is_arabic: bool

    @validator('nationality', pre=True)
    def parse_nationality(cls, v):
        if isinstance(v, str):
            # Normalize string input to the Enum
            try:
                return StudentNationality(v.lower().strip())
            except ValueError:
                return StudentNationality.EGYPTIAN # Fallback
        return v


# --- Search/Content Models ---

class CurriculumContext(BaseModel):
    position: Optional[str] = None
    related_concepts: List[str] = []
    unit_overview: Optional[str] = None
    navigation_hint: Optional[str] = None

class SearchResult(BaseModel):
    id: int
    chunk_text: str
    unit: Optional[str] = None
    concept: Optional[str] = None
    lesson: Optional[str] = None
    distance: float
    curriculum_context: Optional[CurriculumContext] = None
    
    # Allow extra fields from DB that we might not explicitly need yet
    class Config:
        extra = "ignore" 

# --- Interaction Models ---

class AgentRequest(BaseModel):
    user_message: str
    student_id: str
    subject: str = "Science"
    model: str = "gpt-4o" # Default model
    temperature: float = Field(0.3, ge=0.0, le=2.0)
    top_k: int = Field(3, ge=1, le=10)

class ChatMessage(BaseModel):
    role: str
    content: str