import logging
import os
from typing import List, Dict
from fastapi import HTTPException
from openai import OpenAI
import sys
import os
sys.path.append(os.path.abspath(os.path.join(os.path.dirname(__file__), '..')))
from core import StudentNationality

logger = logging.getLogger(__name__)


SYSTEM_PROMPTS: Dict[StudentNationality, str] = {
    StudentNationality.EGYPTIAN: """إنت مدرس كيميا لطفل في ابتدائي. رد باللهجة المصريّة السهلة. كلّم الطفل كأنك بتحكي له بصوت طبيعي. خلي الجمل قصيرة وواضحة، وما تحشرش معلومات كتير في جملة واحدة. ما تقولش الحاجات البديهية اللي هو عارفها زي "المَيَّه بتتشرب". قول المعلومة مرّة واحدة من غير تكرار. لو هتدي مثال أو تشبيه، يكون حاجة جديدة بتوضّح الفكرة، مش مجرد إعادة. خلّي المثال بسيط زي لعبة، شكل، أو صورة في الخيال. اكتب الكلمات زي ما بتنطق بالتشكيل الصح (زي: مَيَّه، أوكسچين). لو فيه رموز كيميائية زي H2O أو CO2 اكتبها زي ما هي. خلي الشرح شبه حكاية صغيرة أو صورة في دماغ الطفل، مش زي شرح كتاب.""",

    StudentNationality.SAUDI: """إنت مُعلّم كيميا لطفل في ابتدائي. رد باللهجة السعوديّة الدارجة والبسيطة. كَلّم الطفل كأنك تحاكيه وجهاً لوجه بصوت طبيعي. خل الجمل قصار وواضحة، لا تكدّس معلومات كثير في جملة وحدة. لا تقول أشياء بديهية يعرفها مثل "المُوَيَّه نشربها". أعط المعلومة مرّة وحدة بلا تكرار. لو بتضرب مثال أو تشبيه، يكون زاوية جديدة توضّح الفكرة، ما يكون تكرار. خلّ المثال شي بسيط يقرّب المعنى للطفل: زي لعبة، حركة، أو صورة يتخيّلها. اكتب الكلمات زي ما تنقال باللهجة وبالتشكيل الصحيح(مثل: مُوَيَّة، هيدروجين، أوكسچين). لو فيه رموز كيميائية مثل H2O أو CO2 اكتُبها زي ما هي. الشرح يكون كأنه سواليف بسيطة أو حكاية تخلي الطفل يتصوّرها، مو زي كلام كتاب مدرسي."""
}


class AgentService:
    """Service class for handling AI agent conversations using OpenAI GPT"""

    def __init__(self):
        self.api_key = os.getenv("OPENAI_API_KEY")
        if not self.api_key:
            logger.warning("Warning: OPENAI_API_KEY not found. Agent service will be disabled.")
            self.client = None
        else:
            self.client = OpenAI(api_key=self.api_key)

        self.conversations: Dict[str, List[Dict[str, str]]] = {}

    def is_available(self) -> bool:
        return self.client is not None

    def get_conversation_history(self, conversation_id: str = "default") -> List[Dict[str, str]]:
        return self.conversations.get(conversation_id, [])

    def add_message_to_history(self, message: str, role: str = "user", conversation_id: str = "default"):
        if conversation_id not in self.conversations:
            self.conversations[conversation_id] = []
        self.conversations[conversation_id].append({"role": role, "content": message})
        if len(self.conversations[conversation_id]) > 20:
            messages = self.conversations[conversation_id]
            if messages[0].get("role") == "system":
                self.conversations[conversation_id] = [messages[0]] + messages[-19:]
            else:
                self.conversations[conversation_id] = messages[-20:]

    def generate_response(
        self,
        user_message: str,
        conversation_id: str = "default",
        model: str = "gpt-5-nano",
        temperature: float = 1.0,
        nationality: StudentNationality = StudentNationality.EGYPTIAN
    ) -> str:
        if not self.is_available():
            raise HTTPException(status_code=500, detail="Agent service not available")

        try:
            self.add_message_to_history(user_message, "user", conversation_id)

            # 🟢 اختر الـ system prompt المناسب للجنسية
            system_prompt = SYSTEM_PROMPTS.get(nationality, SYSTEM_PROMPTS[StudentNationality.EGYPTIAN])

            messages = []
            conversation_history = self.get_conversation_history(conversation_id)
            if not conversation_history or conversation_history[0].get("role") != "system":
                messages.append({"role": "system", "content": system_prompt})
                self.conversations.setdefault(conversation_id, []).insert(0, {
                    "role": "system",
                    "content": system_prompt
                })
            messages.extend(conversation_history)

            response = self.client.chat.completions.create(
                model=model,
                messages=messages,
                temperature=temperature
            )

            ai_response = response.choices[0].message.content.strip()
            if not ai_response:
                raise ValueError("Empty response from AI model")

            self.add_message_to_history(ai_response, "assistant", conversation_id)
            return ai_response

        except Exception as e:
            logger.error(f"Error generating AI response: {e}")
            raise HTTPException(status_code=500, detail=f"AI response generation failed: {str(e)}")


# ----------------- Suggested Test -----------------
if __name__ == "__main__":
    logging.basicConfig(level=logging.INFO)
    agent = AgentService()
    if agent.is_available():
        reply = agent.generate_response("هو يعني إيه ذَرّة؟", model="gpt-5-nano", nationality=StudentNationality.EGYPTIAN)
        print("AI:", reply)
    else:
        print("Agent service not available. Check OPENAI_API_KEY.")
