import os
import psycopg2
from psycopg2.extras import RealDictCursor
from psycopg2.pool import ThreadedConnectionPool
from typing import List, Dict, Optional, Tuple
import logging

logger = logging.getLogger(__name__)


class ChatDatabaseService:
    """Simple service for managing chat history in PostgreSQL with connection pooling"""

    def __init__(self):
        self.pool = ThreadedConnectionPool(
            minconn=1,
            maxconn=20,
            host=os.getenv("POSTGRES_HOST", "postgres"),
            user=os.getenv("POSTGRES_USER"),
            password=os.getenv("POSTGRES_PASSWORD"),
            dbname=os.getenv("POSTGRES_DB"),
        )

    def get_student_nationality(self, student_id: str) -> Optional[str]:
        """Get student nationality from database"""
        conn = self.pool.getconn()
        try:
            with conn.cursor(cursor_factory=RealDictCursor) as cur:
                cur.execute(
                    "SELECT nationality FROM students WHERE student_id = %s",
                    (student_id,)
                )
                result = cur.fetchone()
                return result["nationality"] if result else None
        finally:
            self.pool.putconn(conn)

    def get_student_info(self, student_id: str) -> Optional[Dict]:
        """Get complete student information from database"""
        conn = self.pool.getconn()
        try:
            with conn.cursor(cursor_factory=RealDictCursor) as cur:
                cur.execute(
                    """
                    SELECT student_id, student_name, grade, language, nationality 
                    FROM students 
                    WHERE student_id = %s
                    """,
                    (student_id,)
                )
                result = cur.fetchone()
                if result:
                    return {
                        'student_id': result['student_id'],
                        'student_name': result['student_name'],
                        'grade': result['grade'],  # This is now an integer
                        'is_arabic': result['language'],  # Convert language boolean to is_arabic
                        'nationality': result['nationality']
                    }
                return None
        finally:
            self.pool.putconn(conn)

    def get_student_grade_and_language(self, student_id: str) -> Optional[Tuple[int, bool]]:
        """Get student grade and language preference"""
        conn = self.pool.getconn()
        try:
            with conn.cursor(cursor_factory=RealDictCursor) as cur:
                cur.execute(
                    "SELECT grade, language FROM students WHERE student_id = %s",
                    (student_id,)
                )
                result = cur.fetchone()
                if result:
                    return (result["grade"], result["language"])
                return None
        finally:
            self.pool.putconn(conn)

    def get_chat_history(self, student_id: str, limit: int = 20) -> List[Dict[str, str]]:
        """Get chat history for a student, returns in chronological order"""
        conn = self.pool.getconn()
        try:
            with conn.cursor(cursor_factory=RealDictCursor) as cur:
                cur.execute(
                    """
                    SELECT role, content
                    FROM chat_history
                    WHERE student_id = %s
                    ORDER BY created_at DESC
                    LIMIT %s;
                    """,
                    (student_id, limit)
                )
                results = cur.fetchall()
                # Return in chronological order (oldest first)
                return [{"role": row["role"], "content": row["content"]} for row in reversed(results)]
        finally:
            self.pool.putconn(conn)

    def add_message(self, student_id: str, role: str, content: str):
        """Add a message to chat history"""
        conn = self.pool.getconn()
        try:
            with conn.cursor() as cur:
                cur.execute(
                    """
                    INSERT INTO chat_history (student_id, role, content)
                    VALUES (%s, %s, %s);
                    """,
                    (student_id, role, content)
                )
                conn.commit()
        finally:
            self.pool.putconn(conn)

    def clear_history(self, student_id: str):
        """Clear chat history for a student"""
        conn = self.pool.getconn()
        try:
            with conn.cursor() as cur:
                cur.execute(
                    "DELETE FROM chat_history WHERE student_id = %s",
                    (student_id,)
                )
                conn.commit()
        finally:
            self.pool.putconn(conn)

    def limit_history(self, student_id: str, max_messages: int = 40):
        """Keep only recent messages for a student"""
        conn = self.pool.getconn()
        try:
            with conn.cursor() as cur:
                cur.execute(
                    """
                    DELETE FROM chat_history 
                    WHERE student_id = %s 
                    AND role != 'system'
                    AND id NOT IN (
                        SELECT id FROM chat_history 
                        WHERE student_id = %s AND role != 'system'
                        ORDER BY created_at DESC 
                        LIMIT %s
                    );
                    """,
                    (student_id, student_id, max_messages)
                )
                conn.commit()
        finally:
            self.pool.putconn(conn)

    def update_student_info(self, student_id: str, grade: Optional[int] = None, 
                           language: Optional[bool] = None, nationality: Optional[str] = None):
        """Update student information"""
        updates = []
        params = []
        
        if grade is not None:
            updates.append("grade = %s")
            params.append(grade)
        
        if language is not None:
            updates.append("language = %s")
            params.append(language)
            
        if nationality is not None:
            updates.append("nationality = %s")
            params.append(nationality)
        
        if updates:
            params.append(student_id)
            conn = self.pool.getconn()
            try:
                with conn.cursor() as cur:
                    cur.execute(
                        f"""
                        UPDATE students 
                        SET {', '.join(updates)}
                        WHERE student_id = %s
                        """,
                        params
                    )
                    conn.commit()
            finally:
                self.pool.putconn(conn)

    def create_student(self, student_id: str, student_name: str, grade: int, 
                      language: bool, nationality: str = 'EGYPTIAN'):
        """Create a new student record"""
        conn = self.pool.getconn()
        try:
            with conn.cursor() as cur:
                cur.execute(
                    """
                    INSERT INTO students (student_id, student_name, grade, language, nationality)
                    VALUES (%s, %s, %s, %s, %s)
                    ON CONFLICT (student_id) DO NOTHING;
                    """,
                    (student_id, student_name, grade, language, nationality)
                )
                conn.commit()
        finally:
            self.pool.putconn(conn)

    def close_pool(self):
        if self.pool:
            self.pool.closeall()